﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/ssm-incidents/SSMIncidentsRequest.h>
#include <aws/ssm-incidents/SSMIncidents_EXPORTS.h>
#include <aws/ssm-incidents/model/EventReference.h>

#include <utility>

namespace Aws {
namespace SSMIncidents {
namespace Model {

/**
 */
class UpdateTimelineEventRequest : public SSMIncidentsRequest {
 public:
  AWS_SSMINCIDENTS_API UpdateTimelineEventRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateTimelineEvent"; }

  AWS_SSMINCIDENTS_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>A token that ensures that a client calls the operation only once with the
   * specified details.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  UpdateTimelineEventRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A short description of the event.</p>
   */
  inline const Aws::String& GetEventData() const { return m_eventData; }
  inline bool EventDataHasBeenSet() const { return m_eventDataHasBeenSet; }
  template <typename EventDataT = Aws::String>
  void SetEventData(EventDataT&& value) {
    m_eventDataHasBeenSet = true;
    m_eventData = std::forward<EventDataT>(value);
  }
  template <typename EventDataT = Aws::String>
  UpdateTimelineEventRequest& WithEventData(EventDataT&& value) {
    SetEventData(std::forward<EventDataT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the event to update. You can use <code>ListTimelineEvents</code> to
   * find an event's ID.</p>
   */
  inline const Aws::String& GetEventId() const { return m_eventId; }
  inline bool EventIdHasBeenSet() const { return m_eventIdHasBeenSet; }
  template <typename EventIdT = Aws::String>
  void SetEventId(EventIdT&& value) {
    m_eventIdHasBeenSet = true;
    m_eventId = std::forward<EventIdT>(value);
  }
  template <typename EventIdT = Aws::String>
  UpdateTimelineEventRequest& WithEventId(EventIdT&& value) {
    SetEventId(std::forward<EventIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Updates all existing references in a <code>TimelineEvent</code>. A reference
   * is an Amazon Web Services resource involved or associated with the incident. To
   * specify a reference, enter its Amazon Resource Name (ARN). You can also specify
   * a related item associated with that resource. For example, to specify an Amazon
   * DynamoDB (DynamoDB) table as a resource, use its ARN. You can also specify an
   * Amazon CloudWatch metric associated with the DynamoDB table as a related
   * item.</p>  <p>This update action overrides all existing references.
   * If you want to keep existing references, you must specify them in the call. If
   * you don't, this action removes any existing references and enters only new
   * references.</p>
   */
  inline const Aws::Vector<EventReference>& GetEventReferences() const { return m_eventReferences; }
  inline bool EventReferencesHasBeenSet() const { return m_eventReferencesHasBeenSet; }
  template <typename EventReferencesT = Aws::Vector<EventReference>>
  void SetEventReferences(EventReferencesT&& value) {
    m_eventReferencesHasBeenSet = true;
    m_eventReferences = std::forward<EventReferencesT>(value);
  }
  template <typename EventReferencesT = Aws::Vector<EventReference>>
  UpdateTimelineEventRequest& WithEventReferences(EventReferencesT&& value) {
    SetEventReferences(std::forward<EventReferencesT>(value));
    return *this;
  }
  template <typename EventReferencesT = EventReference>
  UpdateTimelineEventRequest& AddEventReferences(EventReferencesT&& value) {
    m_eventReferencesHasBeenSet = true;
    m_eventReferences.emplace_back(std::forward<EventReferencesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The timestamp for when the event occurred.</p>
   */
  inline const Aws::Utils::DateTime& GetEventTime() const { return m_eventTime; }
  inline bool EventTimeHasBeenSet() const { return m_eventTimeHasBeenSet; }
  template <typename EventTimeT = Aws::Utils::DateTime>
  void SetEventTime(EventTimeT&& value) {
    m_eventTimeHasBeenSet = true;
    m_eventTime = std::forward<EventTimeT>(value);
  }
  template <typename EventTimeT = Aws::Utils::DateTime>
  UpdateTimelineEventRequest& WithEventTime(EventTimeT&& value) {
    SetEventTime(std::forward<EventTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of event. You can update events of type <code>Custom Event</code>
   * and <code>Note</code>.</p>
   */
  inline const Aws::String& GetEventType() const { return m_eventType; }
  inline bool EventTypeHasBeenSet() const { return m_eventTypeHasBeenSet; }
  template <typename EventTypeT = Aws::String>
  void SetEventType(EventTypeT&& value) {
    m_eventTypeHasBeenSet = true;
    m_eventType = std::forward<EventTypeT>(value);
  }
  template <typename EventTypeT = Aws::String>
  UpdateTimelineEventRequest& WithEventType(EventTypeT&& value) {
    SetEventType(std::forward<EventTypeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the incident that includes the timeline
   * event.</p>
   */
  inline const Aws::String& GetIncidentRecordArn() const { return m_incidentRecordArn; }
  inline bool IncidentRecordArnHasBeenSet() const { return m_incidentRecordArnHasBeenSet; }
  template <typename IncidentRecordArnT = Aws::String>
  void SetIncidentRecordArn(IncidentRecordArnT&& value) {
    m_incidentRecordArnHasBeenSet = true;
    m_incidentRecordArn = std::forward<IncidentRecordArnT>(value);
  }
  template <typename IncidentRecordArnT = Aws::String>
  UpdateTimelineEventRequest& WithIncidentRecordArn(IncidentRecordArnT&& value) {
    SetIncidentRecordArn(std::forward<IncidentRecordArnT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};

  Aws::String m_eventData;

  Aws::String m_eventId;

  Aws::Vector<EventReference> m_eventReferences;

  Aws::Utils::DateTime m_eventTime{};

  Aws::String m_eventType;

  Aws::String m_incidentRecordArn;
  bool m_clientTokenHasBeenSet = true;
  bool m_eventDataHasBeenSet = false;
  bool m_eventIdHasBeenSet = false;
  bool m_eventReferencesHasBeenSet = false;
  bool m_eventTimeHasBeenSet = false;
  bool m_eventTypeHasBeenSet = false;
  bool m_incidentRecordArnHasBeenSet = false;
};

}  // namespace Model
}  // namespace SSMIncidents
}  // namespace Aws
