﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/finspace/Finspace_EXPORTS.h>
#include <aws/finspace/model/KxAzMode.h>
#include <aws/finspace/model/KxVolumeStatus.h>
#include <aws/finspace/model/KxVolumeType.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace finspace {
namespace Model {

/**
 * <p> The structure that contains the metadata of the volume. </p><p><h3>See
 * Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/finspace-2021-03-12/KxVolume">AWS
 * API Reference</a></p>
 */
class KxVolume {
 public:
  AWS_FINSPACE_API KxVolume() = default;
  AWS_FINSPACE_API KxVolume(Aws::Utils::Json::JsonView jsonValue);
  AWS_FINSPACE_API KxVolume& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_FINSPACE_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>A unique identifier for the volume.</p>
   */
  inline const Aws::String& GetVolumeName() const { return m_volumeName; }
  inline bool VolumeNameHasBeenSet() const { return m_volumeNameHasBeenSet; }
  template <typename VolumeNameT = Aws::String>
  void SetVolumeName(VolumeNameT&& value) {
    m_volumeNameHasBeenSet = true;
    m_volumeName = std::forward<VolumeNameT>(value);
  }
  template <typename VolumeNameT = Aws::String>
  KxVolume& WithVolumeName(VolumeNameT&& value) {
    SetVolumeName(std::forward<VolumeNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The type of file system volume. Currently, FinSpace only supports
   * <code>NAS_1</code> volume type. </p>
   */
  inline KxVolumeType GetVolumeType() const { return m_volumeType; }
  inline bool VolumeTypeHasBeenSet() const { return m_volumeTypeHasBeenSet; }
  inline void SetVolumeType(KxVolumeType value) {
    m_volumeTypeHasBeenSet = true;
    m_volumeType = value;
  }
  inline KxVolume& WithVolumeType(KxVolumeType value) {
    SetVolumeType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The status of volume.</p> <ul> <li> <p>CREATING – The volume creation is in
   * progress.</p> </li> <li> <p>CREATE_FAILED – The volume creation has failed.</p>
   * </li> <li> <p>ACTIVE – The volume is active.</p> </li> <li> <p>UPDATING – The
   * volume is in the process of being updated.</p> </li> <li> <p>UPDATE_FAILED – The
   * update action failed.</p> </li> <li> <p>UPDATED – The volume is successfully
   * updated.</p> </li> <li> <p>DELETING – The volume is in the process of being
   * deleted.</p> </li> <li> <p>DELETE_FAILED – The system failed to delete the
   * volume.</p> </li> <li> <p>DELETED – The volume is successfully deleted.</p>
   * </li> </ul>
   */
  inline KxVolumeStatus GetStatus() const { return m_status; }
  inline bool StatusHasBeenSet() const { return m_statusHasBeenSet; }
  inline void SetStatus(KxVolumeStatus value) {
    m_statusHasBeenSet = true;
    m_status = value;
  }
  inline KxVolume& WithStatus(KxVolumeStatus value) {
    SetStatus(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> A description of the volume. </p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  KxVolume& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The error message when a failed state occurs. </p>
   */
  inline const Aws::String& GetStatusReason() const { return m_statusReason; }
  inline bool StatusReasonHasBeenSet() const { return m_statusReasonHasBeenSet; }
  template <typename StatusReasonT = Aws::String>
  void SetStatusReason(StatusReasonT&& value) {
    m_statusReasonHasBeenSet = true;
    m_statusReason = std::forward<StatusReasonT>(value);
  }
  template <typename StatusReasonT = Aws::String>
  KxVolume& WithStatusReason(StatusReasonT&& value) {
    SetStatusReason(std::forward<StatusReasonT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The number of availability zones you want to assign per volume. Currently,
   * FinSpace only supports <code>SINGLE</code> for volumes. This places dataview in
   * a single AZ.</p>
   */
  inline KxAzMode GetAzMode() const { return m_azMode; }
  inline bool AzModeHasBeenSet() const { return m_azModeHasBeenSet; }
  inline void SetAzMode(KxAzMode value) {
    m_azModeHasBeenSet = true;
    m_azMode = value;
  }
  inline KxVolume& WithAzMode(KxAzMode value) {
    SetAzMode(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The identifier of the availability zones.</p>
   */
  inline const Aws::Vector<Aws::String>& GetAvailabilityZoneIds() const { return m_availabilityZoneIds; }
  inline bool AvailabilityZoneIdsHasBeenSet() const { return m_availabilityZoneIdsHasBeenSet; }
  template <typename AvailabilityZoneIdsT = Aws::Vector<Aws::String>>
  void SetAvailabilityZoneIds(AvailabilityZoneIdsT&& value) {
    m_availabilityZoneIdsHasBeenSet = true;
    m_availabilityZoneIds = std::forward<AvailabilityZoneIdsT>(value);
  }
  template <typename AvailabilityZoneIdsT = Aws::Vector<Aws::String>>
  KxVolume& WithAvailabilityZoneIds(AvailabilityZoneIdsT&& value) {
    SetAvailabilityZoneIds(std::forward<AvailabilityZoneIdsT>(value));
    return *this;
  }
  template <typename AvailabilityZoneIdsT = Aws::String>
  KxVolume& AddAvailabilityZoneIds(AvailabilityZoneIdsT&& value) {
    m_availabilityZoneIdsHasBeenSet = true;
    m_availabilityZoneIds.emplace_back(std::forward<AvailabilityZoneIdsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The timestamp at which the volume was created in FinSpace. The value is
   * determined as epoch time in milliseconds. For example, the value for Monday,
   * November 1, 2021 12:00:00 PM UTC is specified as 1635768000000.</p>
   */
  inline const Aws::Utils::DateTime& GetCreatedTimestamp() const { return m_createdTimestamp; }
  inline bool CreatedTimestampHasBeenSet() const { return m_createdTimestampHasBeenSet; }
  template <typename CreatedTimestampT = Aws::Utils::DateTime>
  void SetCreatedTimestamp(CreatedTimestampT&& value) {
    m_createdTimestampHasBeenSet = true;
    m_createdTimestamp = std::forward<CreatedTimestampT>(value);
  }
  template <typename CreatedTimestampT = Aws::Utils::DateTime>
  KxVolume& WithCreatedTimestamp(CreatedTimestampT&& value) {
    SetCreatedTimestamp(std::forward<CreatedTimestampT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The last time that the volume was updated in FinSpace. The value is
   * determined as epoch time in milliseconds. For example, the value for Monday,
   * November 1, 2021 12:00:00 PM UTC is specified as 1635768000000.</p>
   */
  inline const Aws::Utils::DateTime& GetLastModifiedTimestamp() const { return m_lastModifiedTimestamp; }
  inline bool LastModifiedTimestampHasBeenSet() const { return m_lastModifiedTimestampHasBeenSet; }
  template <typename LastModifiedTimestampT = Aws::Utils::DateTime>
  void SetLastModifiedTimestamp(LastModifiedTimestampT&& value) {
    m_lastModifiedTimestampHasBeenSet = true;
    m_lastModifiedTimestamp = std::forward<LastModifiedTimestampT>(value);
  }
  template <typename LastModifiedTimestampT = Aws::Utils::DateTime>
  KxVolume& WithLastModifiedTimestamp(LastModifiedTimestampT&& value) {
    SetLastModifiedTimestamp(std::forward<LastModifiedTimestampT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_volumeName;

  KxVolumeType m_volumeType{KxVolumeType::NOT_SET};

  KxVolumeStatus m_status{KxVolumeStatus::NOT_SET};

  Aws::String m_description;

  Aws::String m_statusReason;

  KxAzMode m_azMode{KxAzMode::NOT_SET};

  Aws::Vector<Aws::String> m_availabilityZoneIds;

  Aws::Utils::DateTime m_createdTimestamp{};

  Aws::Utils::DateTime m_lastModifiedTimestamp{};
  bool m_volumeNameHasBeenSet = false;
  bool m_volumeTypeHasBeenSet = false;
  bool m_statusHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_statusReasonHasBeenSet = false;
  bool m_azModeHasBeenSet = false;
  bool m_availabilityZoneIdsHasBeenSet = false;
  bool m_createdTimestampHasBeenSet = false;
  bool m_lastModifiedTimestampHasBeenSet = false;
};

}  // namespace Model
}  // namespace finspace
}  // namespace Aws
