﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/migrationhuborchestrator/MigrationHubOrchestrator_EXPORTS.h>
#include <aws/migrationhuborchestrator/model/TemplateInput.h>
#include <aws/migrationhuborchestrator/model/TemplateStatus.h>
#include <aws/migrationhuborchestrator/model/Tool.h>

#include <utility>

namespace Aws {
template <typename RESULT_TYPE>
class AmazonWebServiceResult;

namespace Utils {
namespace Json {
class JsonValue;
}  // namespace Json
}  // namespace Utils
namespace MigrationHubOrchestrator {
namespace Model {
class GetTemplateResult {
 public:
  AWS_MIGRATIONHUBORCHESTRATOR_API GetTemplateResult() = default;
  AWS_MIGRATIONHUBORCHESTRATOR_API GetTemplateResult(const Aws::AmazonWebServiceResult<Aws::Utils::Json::JsonValue>& result);
  AWS_MIGRATIONHUBORCHESTRATOR_API GetTemplateResult& operator=(const Aws::AmazonWebServiceResult<Aws::Utils::Json::JsonValue>& result);

  ///@{
  /**
   * <p>The ID of the template.</p>
   */
  inline const Aws::String& GetId() const { return m_id; }
  template <typename IdT = Aws::String>
  void SetId(IdT&& value) {
    m_idHasBeenSet = true;
    m_id = std::forward<IdT>(value);
  }
  template <typename IdT = Aws::String>
  GetTemplateResult& WithId(IdT&& value) {
    SetId(std::forward<IdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>&gt;The Amazon Resource Name (ARN) of the migration workflow template. The
   * format for an Migration Hub Orchestrator template ARN is
   * <code>arn:aws:migrationhub-orchestrator:region:account:template/template-abcd1234</code>.
   * For more information about ARNs, see <a
   * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/reference-arns.html">Amazon
   * Resource Names (ARNs)</a> in the <i>AWS General Reference</i>.</p>
   */
  inline const Aws::String& GetTemplateArn() const { return m_templateArn; }
  template <typename TemplateArnT = Aws::String>
  void SetTemplateArn(TemplateArnT&& value) {
    m_templateArnHasBeenSet = true;
    m_templateArn = std::forward<TemplateArnT>(value);
  }
  template <typename TemplateArnT = Aws::String>
  GetTemplateResult& WithTemplateArn(TemplateArnT&& value) {
    SetTemplateArn(std::forward<TemplateArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the template.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  GetTemplateResult& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The time at which the template was last created.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  GetTemplateResult& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The inputs provided for the creation of the migration workflow.</p>
   */
  inline const Aws::Vector<TemplateInput>& GetInputs() const { return m_inputs; }
  template <typename InputsT = Aws::Vector<TemplateInput>>
  void SetInputs(InputsT&& value) {
    m_inputsHasBeenSet = true;
    m_inputs = std::forward<InputsT>(value);
  }
  template <typename InputsT = Aws::Vector<TemplateInput>>
  GetTemplateResult& WithInputs(InputsT&& value) {
    SetInputs(std::forward<InputsT>(value));
    return *this;
  }
  template <typename InputsT = TemplateInput>
  GetTemplateResult& AddInputs(InputsT&& value) {
    m_inputsHasBeenSet = true;
    m_inputs.emplace_back(std::forward<InputsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>List of AWS services utilized in a migration workflow.</p>
   */
  inline const Aws::Vector<Tool>& GetTools() const { return m_tools; }
  template <typename ToolsT = Aws::Vector<Tool>>
  void SetTools(ToolsT&& value) {
    m_toolsHasBeenSet = true;
    m_tools = std::forward<ToolsT>(value);
  }
  template <typename ToolsT = Aws::Vector<Tool>>
  GetTemplateResult& WithTools(ToolsT&& value) {
    SetTools(std::forward<ToolsT>(value));
    return *this;
  }
  template <typename ToolsT = Tool>
  GetTemplateResult& AddTools(ToolsT&& value) {
    m_toolsHasBeenSet = true;
    m_tools.emplace_back(std::forward<ToolsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The time at which the template was last created.</p>
   */
  inline const Aws::Utils::DateTime& GetCreationTime() const { return m_creationTime; }
  template <typename CreationTimeT = Aws::Utils::DateTime>
  void SetCreationTime(CreationTimeT&& value) {
    m_creationTimeHasBeenSet = true;
    m_creationTime = std::forward<CreationTimeT>(value);
  }
  template <typename CreationTimeT = Aws::Utils::DateTime>
  GetTemplateResult& WithCreationTime(CreationTimeT&& value) {
    SetCreationTime(std::forward<CreationTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The owner of the migration workflow template.</p>
   */
  inline const Aws::String& GetOwner() const { return m_owner; }
  template <typename OwnerT = Aws::String>
  void SetOwner(OwnerT&& value) {
    m_ownerHasBeenSet = true;
    m_owner = std::forward<OwnerT>(value);
  }
  template <typename OwnerT = Aws::String>
  GetTemplateResult& WithOwner(OwnerT&& value) {
    SetOwner(std::forward<OwnerT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The status of the template.</p>
   */
  inline TemplateStatus GetStatus() const { return m_status; }
  inline void SetStatus(TemplateStatus value) {
    m_statusHasBeenSet = true;
    m_status = value;
  }
  inline GetTemplateResult& WithStatus(TemplateStatus value) {
    SetStatus(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The status message of retrieving migration workflow templates.</p>
   */
  inline const Aws::String& GetStatusMessage() const { return m_statusMessage; }
  template <typename StatusMessageT = Aws::String>
  void SetStatusMessage(StatusMessageT&& value) {
    m_statusMessageHasBeenSet = true;
    m_statusMessage = std::forward<StatusMessageT>(value);
  }
  template <typename StatusMessageT = Aws::String>
  GetTemplateResult& WithStatusMessage(StatusMessageT&& value) {
    SetStatusMessage(std::forward<StatusMessageT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The class of the migration workflow template. The available template classes
   * are:</p> <ul> <li> <p>A2C</p> </li> <li> <p>MGN</p> </li> <li> <p>SAP_MULTI</p>
   * </li> <li> <p>SQL_EC2</p> </li> <li> <p>SQL_RDS</p> </li> <li> <p>VMIE</p> </li>
   * </ul>
   */
  inline const Aws::String& GetTemplateClass() const { return m_templateClass; }
  template <typename TemplateClassT = Aws::String>
  void SetTemplateClass(TemplateClassT&& value) {
    m_templateClassHasBeenSet = true;
    m_templateClass = std::forward<TemplateClassT>(value);
  }
  template <typename TemplateClassT = Aws::String>
  GetTemplateResult& WithTemplateClass(TemplateClassT&& value) {
    SetTemplateClass(std::forward<TemplateClassT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The tags added to the migration workflow template.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  GetTemplateResult& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  GetTemplateResult& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}

  ///@{

  inline const Aws::String& GetRequestId() const { return m_requestId; }
  template <typename RequestIdT = Aws::String>
  void SetRequestId(RequestIdT&& value) {
    m_requestIdHasBeenSet = true;
    m_requestId = std::forward<RequestIdT>(value);
  }
  template <typename RequestIdT = Aws::String>
  GetTemplateResult& WithRequestId(RequestIdT&& value) {
    SetRequestId(std::forward<RequestIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_id;

  Aws::String m_templateArn;

  Aws::String m_name;

  Aws::String m_description;

  Aws::Vector<TemplateInput> m_inputs;

  Aws::Vector<Tool> m_tools;

  Aws::Utils::DateTime m_creationTime{};

  Aws::String m_owner;

  TemplateStatus m_status{TemplateStatus::NOT_SET};

  Aws::String m_statusMessage;

  Aws::String m_templateClass;

  Aws::Map<Aws::String, Aws::String> m_tags;

  Aws::String m_requestId;
  bool m_idHasBeenSet = false;
  bool m_templateArnHasBeenSet = false;
  bool m_nameHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_inputsHasBeenSet = false;
  bool m_toolsHasBeenSet = false;
  bool m_creationTimeHasBeenSet = false;
  bool m_ownerHasBeenSet = false;
  bool m_statusHasBeenSet = false;
  bool m_statusMessageHasBeenSet = false;
  bool m_templateClassHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
  bool m_requestIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace MigrationHubOrchestrator
}  // namespace Aws
