﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/apigateway/APIGatewayRequest.h>
#include <aws/apigateway/APIGateway_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>

#include <utility>

namespace Aws {
namespace APIGateway {
namespace Model {

/**
 * <p>Creates a VPC link, under the caller's account in a selected region, in an
 * asynchronous operation that typically takes 2-4 minutes to complete and become
 * operational. The caller must have permissions to create and update VPC Endpoint
 * services.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/apigateway-2015-07-09/CreateVpcLinkRequest">AWS
 * API Reference</a></p>
 */
class CreateVpcLinkRequest : public APIGatewayRequest {
 public:
  AWS_APIGATEWAY_API CreateVpcLinkRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateVpcLink"; }

  AWS_APIGATEWAY_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The name used to label and identify the VPC link.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreateVpcLinkRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The description of the VPC link.</p>
   */
  inline const Aws::String& GetDescription() const { return m_description; }
  inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
  template <typename DescriptionT = Aws::String>
  void SetDescription(DescriptionT&& value) {
    m_descriptionHasBeenSet = true;
    m_description = std::forward<DescriptionT>(value);
  }
  template <typename DescriptionT = Aws::String>
  CreateVpcLinkRequest& WithDescription(DescriptionT&& value) {
    SetDescription(std::forward<DescriptionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of the network load balancer of the VPC targeted by the VPC link. The
   * network load balancer must be owned by the same Amazon Web Services account of
   * the API owner.</p>
   */
  inline const Aws::Vector<Aws::String>& GetTargetArns() const { return m_targetArns; }
  inline bool TargetArnsHasBeenSet() const { return m_targetArnsHasBeenSet; }
  template <typename TargetArnsT = Aws::Vector<Aws::String>>
  void SetTargetArns(TargetArnsT&& value) {
    m_targetArnsHasBeenSet = true;
    m_targetArns = std::forward<TargetArnsT>(value);
  }
  template <typename TargetArnsT = Aws::Vector<Aws::String>>
  CreateVpcLinkRequest& WithTargetArns(TargetArnsT&& value) {
    SetTargetArns(std::forward<TargetArnsT>(value));
    return *this;
  }
  template <typename TargetArnsT = Aws::String>
  CreateVpcLinkRequest& AddTargetArns(TargetArnsT&& value) {
    m_targetArnsHasBeenSet = true;
    m_targetArns.emplace_back(std::forward<TargetArnsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The key-value map of strings. The valid character set is [a-zA-Z+-=._:/]. The
   * tag key can be up to 128 characters and must not start with <code>aws:</code>.
   * The tag value can be up to 256 characters.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  CreateVpcLinkRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  CreateVpcLinkRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_name;

  Aws::String m_description;

  Aws::Vector<Aws::String> m_targetArns;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_nameHasBeenSet = false;
  bool m_descriptionHasBeenSet = false;
  bool m_targetArnsHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace APIGateway
}  // namespace Aws
